#region References

using System;
using System.Xml;
using System.Text;
using System.Data;
using System.Collections;

#endregion

namespace gov.va.med.vbecs.DAL.HL7.OpenLibrary
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>Brian    lin</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>10/20/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	// <summary>
	// This class provides utility methods for HL7 parsing. It performs necessary checks 
	// while retrieving HL7 nodes and throws HL7Exception if needed.
	// It must be used for HL7 objects serialization / deserialization / parsing only.
	// </summary>

	#endregion

	/// <summary>
	/// 
	/// </summary>
	public sealed class HL7Utility
	{
		/// <summary>
		/// This class shouldn't be initialized. It only provides a number of static utility methods
		/// helping with HL7 parsing / deserialization.
		/// </summary>
		private HL7Utility() {}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5724"> 
		///		<ExpectedInput>Valid HL7 PID segment</ExpectedInput>
		///		<ExpectedOutput>PID string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5725"> 
		///		<ExpectedInput>Null hl7Segment</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5726"> 
		///		<ExpectedInput>hl7Segment input parameter has less than 3 characters.</ExpectedInput>
		///		<ExpectedOutput>null</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the segment ID of a segment
		/// </summary>
		/// <param name="hl7Segment">HL7 Segment strign</param>
		/// <returns>returns the segment id (first 3 characters) of an HL7 segment string</returns>
		public static string ParseGetSegmentID( string hl7Segment )
		{
			if ( hl7Segment == null )
				throw( new ArgumentNullException( "hl7Segment" ) );
			if ( hl7Segment.Length < 3 )
				return null;

			return hl7Segment.Substring( 0,3 );
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5727"> 
		///		<ExpectedInput>Non-null Acknowledgement message.</ExpectedInput>
		///		<ExpectedOutput>2 character acknowledgement code of either "AA", "AE", or "AR".</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5728"> 
		///		<ExpectedInput>null Acknowledgement message.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5790"> 
		///		<ExpectedInput>HL7 Message that does not contain an MSA segment.</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the Acknowledgement code of an HL7OrgMessage
		/// </summary>
		/// <param name="msg">HL7 Acknowledgement message string</param>
		/// <returns>2 character Acknowledgement code</returns>
		public static string ParseGetAckCode( string msg )
		{
			if ( msg == null )
				throw( new ArgumentNullException( "msg" ) );

			char[] delimiters = ParseGetMessageDelimiters( msg );

			// if the message does not contain an MSA segment an HL7Exception is thrown;
			string msaSegment = ParseGetRequiredMessageSegment( msg, SegmentTypeNames.MSA );

			return ParseGetSegmentData( msaSegment, delimiters[0].ToString(), 1);
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5729"> 
		///		<ExpectedInput>Non-null HL7 message string</ExpectedInput>
		///		<ExpectedOutput>Message Type and Trigger Event from MSH segment separated by "^".</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5730"> 
		///		<ExpectedInput>Null msg input string.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the Message Type and Trigger Event of the message to determine which
		/// HL7ProtocolMessage fits.
		/// CR 2961
		/// </summary>
		/// <param name="msg">HL7 Message string</param>
		/// <returns>Message Type and Trigger Event separated by a delimiters[1]</returns>
		public static string ParseGetHL7MessageTypeAndTriggerEventString( string msg )
		{
			if ( msg == null )
			{
				throw( new ArgumentNullException( "msg" ) );
			}
			//
			char[] delimiters = ParseGetMessageDelimiters( msg );
			//
			// Split on the \r's to reveal the segments
			string[] hl7Segments = msg.Split(new char[]{'\r'});
			//
			// Split the MSH segment to get control fields
			string[] msh = hl7Segments[0].Split(delimiters[0]);
			//
			string[] msgType = msh[8].Split(delimiters[1]);
			//
			// CR 2961
			return msgType != null && msgType.Length == 2 ? string.Concat(msgType[0], delimiters[1], msgType[1]) : string.Empty;
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/28/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8887"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8888"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the Message Type and Trigger Event of the message to determine which
		/// HL7ProtocolMessage fits.
		/// CR 2961
		/// </summary>
		/// <param name="msg">HL7 Message string</param>
		/// <returns>Message Type and Trigger Event separated by a delimiters[1]</returns>
		public static string [] ParseGetHL7MessageTypeAndTriggerEventArray( string msg )
		{
			if ( msg == null )
			{
				throw( new ArgumentNullException( "msg" ) );
			}
			//
			char[] delimiters = ParseGetMessageDelimiters( msg );
			//
			// Split on the \r's to reveal the segments
			string[] hl7Segments = msg.Split(new char[]{'\r'});
			//
			// Split the MSH segment to get control fields
			string[] msh = hl7Segments[0].Split(delimiters[0]);
			//
			return msh[8].Split(delimiters[1]);
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5731"> 
		///		<ExpectedInput>Non-null HL7 message string</ExpectedInput>
		///		<ExpectedOutput>Message Control ID from the MSH segment</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5732"> 
		///		<ExpectedInput>Null msg input string.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the Message Control ID from the MSH segmetn of a valid HL7 Message.
		/// </summary>
		/// <param name="msg">HL7 Message string</param>
		/// <returns>Message Control ID</returns>
		public static string GetMessageControlID( string msg )
		{
			if ( msg == null )
				throw( new ArgumentNullException( "msg" ) );
			
			char[] delimiters = ParseGetMessageDelimiters( msg );

			//Split on the \r's to reveal the segments
			string[] hl7Segments = msg.Split(new char[]{'\r'});

			//Split the MSH segment to get control fields
			string[] msh = hl7Segments[0].Split(delimiters[0]);

			return msh[9].ToString();
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5733"> 
		///		<ExpectedInput>Non-null HL7 message string</ExpectedInput>
		///		<ExpectedOutput>Character array containing 5 elements.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5734"> 
		///		<ExpectedInput>Null msg input string.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5735"> 
		///		<ExpectedInput>Invalid HL7 Message missing the MSH segment identifiers.</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the HL7 Field Separator and Encoding Characters for a valid HL7 message.
		/// </summary>
		/// <param name="msg">HL7 Message string</param>
		/// <returns>Char array of HL7 message delimiters</returns>
		public static char[] ParseGetMessageDelimiters( string msg )
		{
			if ( msg == null )
				throw( new ArgumentNullException( "msg" ) );
			if ( msg.Substring(0, 3) != "MSH" )
				throw( new ArgumentException( "Not MSH segment" ) );

			return msg.ToCharArray(3, 5);
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5736"> 
		///		<ExpectedInput>Non-null HL7 message string</ExpectedInput>
		///		<ExpectedOutput>Non-null String array containing each segment in the HL7 message.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5737"> 
		///		<ExpectedInput>Null msg input string.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets all of the segments in a valid HL7 message separated by carriage returns.
		/// </summary>
		/// <param name="msg">HL7 Message string</param>
		/// <returns>string array of HL7 message segment</returns>
		public static string[] ParseGetAllMessageSegments( string msg )
		{
			if ( msg == null )
				throw( new ArgumentNullException( "msg" ) );

			return msg.Split(new char[]{'\r'});
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5738"> 
		///		<ExpectedInput>Non-null HL7 message string and valid MSH segment type.</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7 message segment</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5739"> 
		///		<ExpectedInput>Null msg input string.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5955"> 
		///		<ExpectedInput>Non-null HL7 message string and invalid MGR segment type.</ExpectedInput>
		///		<ExpectedOutput>null</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets an option message segment or returns null if the segment doesn't exist in the message.
		/// </summary>
		/// <param name="msg">HL7 Message string</param>
		/// <param name="id">Valid HL7 segment id defined in SegmentTypeNames enum</param>
		/// <returns>HL7 Message Segment string identified by the SegmentTypeName.</returns>
		public static string ParseGetOptionalMessageSegment( string msg, SegmentTypeNames id )
		{
			if ( msg == null )
				throw( new ArgumentNullException( "msg" ) );
			try
			{
				return ParseGetRequiredMessageSegment( msg, id );
			}
			catch( HL7Exception )
			{
				return null;
			}

		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5740"> 
		///		<ExpectedInput>Non-null HL7 message string</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7 message segment</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5741"> 
		///		<ExpectedInput>Null msg input string</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5827"> 
		///		<ExpectedInput>Non-null HL7 message string missing MSA segment with SegmentTypeNames.MSA requested.</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks if a required segment exists and returns a string representing the segment.
		/// </summary>
		/// <param name="msg">HL7 Message string</param>
		/// <param name="id">Valid HL7 segment id defined in SegmentTypeNames enum</param>
		/// <returns>HL7 Message Segment string identified by the SegmentTypeName.</returns>
		public static string ParseGetRequiredMessageSegment( string msg, SegmentTypeNames id )
		{
			if ( msg == null )
				throw( new HL7Exception( "HL7Utility.ParseGetRequiredMessageSegment was called with a null message string" ) );
			
			
			string[] _segments = ParseGetAllMessageSegments( msg );
			
			for( int i=0; i <= _segments.Length -1; i++ )
			{
				if ( _segments[i] != null && _segments[i].Length > 3 )
				{
					if ( _segments[i].Substring(0, 3) == id.ToString().ToUpper() )
					{
						return ConvertString( _segments[i] );
					}
				}
			}
			throw( new HL7Exception( id.ToString() + " segment not found" ) );

		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5742"> 
		///		<ExpectedInput>Non-null HL7 message segment string, valid HL7 encoding character, valid position that holds data.</ExpectedInput>
		///		<ExpectedOutput>Non-null string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5743"> 
		///		<ExpectedInput>Null segment input string</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5744"> 
		///		<ExpectedInput>Null delimiter string</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5745"> 
		///		<ExpectedInput>Position parameter is greater than the length of the segment.</ExpectedInput>
		///		<ExpectedOutput>ArgumentOutOfRangeException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Parses a message segment string and returns the value of a position in the string
		/// </summary>
		/// <param name="segment">HL7 segment</param>
		/// <param name="delimiter">HL7 encoding or field seprator character</param>
		/// <param name="position">Position in segment to return data starting at 0.</param>
		/// <returns>Data within a specific location in an HL7 message segment.</returns>
		public static string ParseGetSegmentData( string segment, string delimiter, int position )
		{
			if ( segment == null )
				throw( new ArgumentNullException( "segment" ) );
			if ( delimiter == null )
				throw( new ArgumentNullException( "delimiter" ) );
			if ( position > segment.Length || position < 0 )
				throw( new ArgumentOutOfRangeException( "position" ) );

			string[] retVal = ParseGetStringArray( segment, delimiter);
			if( retVal.Length <= position )
				return null;

			return ConvertString( retVal[position] );
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5795"> 
		///		<ExpectedInput>Valid HL7 String message</ExpectedInput>
		///		<ExpectedOutput>Non-null String Array.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5796"> 
		///		<ExpectedInput>Null msg string input parameter</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5797"> 
		///		<ExpectedInput>Null delimiter string</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static method to return a string array of elements in a string split by the specified delimiter.
		/// </summary>
		/// <param name="msg">Valid HL7 Message string</param>
		/// <param name="delimiter">Delimiter used to split the message string.</param>
		/// <returns>String Array containing delimited fields in the message string.</returns>
		public static string[] ParseGetStringArray( string msg, string delimiter )
		{
			if ( msg == null )
				throw( new ArgumentNullException( "msg" ) );
			if ( delimiter == null )
				throw( new ArgumentNullException( "delimiter" ) );

			char[] delim = delimiter.ToCharArray();
			return msg.Split(delim[0]);
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5746"> 
		///		<ExpectedInput>Non-null HL7 message string containing order comments in the NTE segment.</ExpectedInput>
		///		<ExpectedOutput>Non-null ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5747"> 
		///		<ExpectedInput>Null msg input string</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Parses message and returns an ArrayList of Order Comments found in the NTE segments
		/// </summary>
		/// <param name="msg">Valid HL7 message string</param>
		/// <returns><see cref="ArrayList"/> containing order comments.</returns>
		public static ArrayList ParseGetOrderComments( string msg )
		{
			if ( msg == null )
				throw( new ArgumentNullException( "msg" ) );
			string[] segments = ParseGetAllMessageSegments( msg );
			ArrayList nteSegments = new ArrayList();
			for( int i = 0; i < segments.Length; i++ )
			{
				if( ParseGetSegmentID( segments[i] ) == "NTE" )
				{
					nteSegments.Add( segments[i] );
				}
			}
			return nteSegments;
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="1" testid ="5748"> 
		///		<ExpectedInput>Empty string</ExpectedInput>
		///		<ExpectedOutput>Null</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5749"> 
		///		<ExpectedInput>valid string</ExpectedInput>
		///		<ExpectedOutput>same valid string</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Converts an empty string to null or returns the string.
		/// </summary>
		/// <param name="inputStr">input string to validate</param>
		/// <returns>either null or the input string</returns>
		public static string ConvertString(string inputStr)
		{
			return inputStr == string.Empty ? null : inputStr;
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6967"> 
		///		<ExpectedInput>System.Exception with message text of "Exception Unit Test".</ExpectedInput>
		///		<ExpectedOutput>string = "Exception Unit Test" + new line</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6968"> 
		///		<ExpectedInput>Null Exception input parameter</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6969"> 
		///		<ExpectedInput>System.Exception with message text of "Exception Unit Test" and
		///		and InnerException with the message text of "Inner Exception Text"</ExpectedInput>
		///		<ExpectedOutput>string = "Exception Unit Test" + new line + "Inner Exception Text" + new line</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Static method to concatenate an inner exception text with the outer exception text.
		/// </summary>
		/// <param name="e">System.Exception</param>
		/// <returns>String containing Message text from the Exception concatenated
		/// with the InnerException Message text.</returns>
		public static string GetInnerExceptionFromException(Exception e)
		{
			if(e == null)
				throw( new ArgumentNullException("e"));

			string errorMessage = "";
			Exception curException = e;
			while( curException != null )
			{
				// CR 2961
				errorMessage = string.Concat(errorMessage, curException.Message, "\n");
				curException = curException.InnerException;
			}
			return errorMessage;
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/23/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8744"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8745"> 
		///		<ExpectedInput>null</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2961
		/// </summary>
		public static string [] ParseGetErrorCodeAndMessage( string msg )
		{
			if ( msg == null )
				throw( new ArgumentNullException( "msg" ) );

			char[] delimiters = ParseGetMessageDelimiters( msg );

			string errSegment = ParseGetOptionalMessageSegment( msg, SegmentTypeNames.ERR );

			if ( errSegment != null )
			{
				return ParseGetSegmentData( errSegment, delimiters[0].ToString(), 3).Split(delimiters[1]);
			}
			else
			{
				return null;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/29/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8781"> 
		///		<ExpectedInput>segment and delimiter string</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8782"> 
		///		<ExpectedInput>null inputs</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns VistaPatientId, PatientIcn, PatientSsn
		/// CR 2960
		/// CR 2971: Renamed from ParseGetPatientIdArray(), 
		/// changed to return only Patient DFN (VistAPatientId)
		/// </summary>
		public static long ParseGetVistAPatientId( string pidSegment, char [] encodingCharacters )
		{
			if ( pidSegment == null )
			{
				throw( new ArgumentNullException( "pidSegment" ) );
			}
			//
			if ( encodingCharacters == null )
			{
				throw( new ArgumentNullException( "encodingCharacters" ) );
			}
			//
			string [] pidValues = pidSegment.Split( encodingCharacters[0] );
			//
			string pid3 = pidValues[3].Split( encodingCharacters[1] )[0];
			//
			return Convert.ToInt64( pid3 );
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8859"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8860"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2985. CR 3243
		/// </summary>
		public static int GetAckErrorCodeFromEnum( AckErrorCodes ackErrorCode )
		{
			switch( ackErrorCode )
			{
				case AckErrorCodes.SegmentSequenceError		: return 100;
				case AckErrorCodes.RequiredFieldMissing		: return 101;
				case AckErrorCodes.DataTypeError			: return 102;
				case AckErrorCodes.TableValueNotFound		: return 103;
				case AckErrorCodes.UnsupportedMessageType	: return 200;
				case AckErrorCodes.UnsupportedEventCode		: return 201;
				case AckErrorCodes.UnsupportedProcessingId	: return 202;
				case AckErrorCodes.UnsupportedVersionId		: return 203;
				case AckErrorCodes.UnknownKeyIdentifier		: return 204;
				case AckErrorCodes.DuplicateKeyIdentifier	: return 205;
				case AckErrorCodes.ApplicationRecordLocked	: return 206;
				case AckErrorCodes.ApplicationInternalError	: return 207;
                case AckErrorCodes.ConflictingProcessingId  : return 208;
				default										: return 0;		// AckErrorCodes.MessageAccepted
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/26/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8861"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8862"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2985
		/// </summary>
		public static char GetAckErrorSeverityFromEnum( AckErrorSeverity ackErrorSeverity )
		{
			switch( ackErrorSeverity )
			{
				case AckErrorSeverity.Information	: return 'I';
				case AckErrorSeverity.Warning		: return 'W';
				default								: return 'E';		// AckErrorSeverity.Error
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/21/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8922"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8923"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2985
		/// </summary>
		public static BloodProductCommentType GetBloodProductCommentTypeEnumFromString( string commentType )
		{
			switch( commentType )
			{
				case "VA-BCR"								: return BloodProductCommentType.Barcode;
				case "RE"									: return BloodProductCommentType.FreeText;
				default										: return BloodProductCommentType.Unknown;	
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/21/2009</CreationDate>
		///<TestCases> 
		///	
		///<Case type="0" testid ="8924"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8925"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2985
		/// </summary>
		public static BloodProductMessageStatus GetBloodProductMessageStatusEnumFromChar( char messageStatus )
		{
			switch( messageStatus )
			{
				case 'F'									: return BloodProductMessageStatus.Final;
				case 'P'									: return BloodProductMessageStatus.Preliminary;
				default										: return BloodProductMessageStatus.Unknown;	
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/21/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8926"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8927"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2985
		/// </summary>
		public static BloodProductTransfusionStatus GetBloodProductTransfusionStatusEnumFromString( string transfusionStatus )
		{
			switch( transfusionStatus )
			{
				case "RA"									: return BloodProductTransfusionStatus.RA;
				case "RL"									: return BloodProductTransfusionStatus.RL;
				case "TR"									: return BloodProductTransfusionStatus.TR;
				case "TX"									: return BloodProductTransfusionStatus.TX;
				case "WA"									: return BloodProductTransfusionStatus.WA;
				default										: return BloodProductTransfusionStatus.UK;	
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/7/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8930"> 
		///		<ExpectedInput>Valid HL7Message</ExpectedInput>
		///		<ExpectedOutput>Patient Names</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8931"> 
		///		<ExpectedInput>Invalid HL7Message</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2606
		/// </summary>
		/// <param name="hl7Message">HL7 message</param>
		/// <param name="patientLastName"></param>
		/// <param name="patientFirstName"></param>
		/// <param name="patientMiddleName"></param>
		public static void GetPatientNameFields( string hl7Message, ref string patientLastName, ref string patientFirstName, ref string patientMiddleName )
		{
			if ( hl7Message == null )
			{
				throw( new ArgumentNullException( "hl7Message" ) );
			}
			//
			char [] delimiters = hl7Message.ToCharArray( 3, 5 );
			//
			string pidSegment = HL7Utility.ParseGetRequiredMessageSegment( hl7Message, SegmentTypeNames.PID );
			//
			string [] patientName = pidSegment.Split( delimiters[0] )[5].Split( delimiters[1] ) ;
			//
			if ( patientName != null )
			{
				patientFirstName = patientName.Length >= 2 ? HL7Utility.ConvertString( patientName[1] ) : null;
				patientLastName = patientName.Length >= 2 ? HL7Utility.ConvertString( patientName[0] ) : null;
				patientMiddleName = patientName.Length >= 3 ? HL7Utility.ConvertString( patientName[2] ) : null;
			}
		}

	}
}
